<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
// use Stripe;
use Illuminate\View\View;
use Illuminate\Http\JsonResponse;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use Illuminate\Support\Facades\Crypt;
use App\Models\User;
use App\Models\Plan;
use App\Models\Payment;
use App\Models\Invoice;

class StripePaymentController extends Controller
{
    /**
     * Show the payment form.
     *
     * @return \Illuminate\View\View
     */
    public function stripe(): View
    {
        return view('stripe');
    }
    
    /**
     * Handle the payment submission.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function stripePost(Request $request): JsonResponse
    {
 

        Stripe::setApiKey(env('STRIPE_SECRET'));

try {

    // plain id 
    $id = Crypt::decrypt($request->id);
    $plan = Plan::find($id);

    $paymentIntent = PaymentIntent::create([
        'amount' => $plan->amount * 100,
        'currency' => 'usd',
        'description' => 'Test payment from developer',
        'payment_method' => $request->stripePaymentMethodId,
        'confirmation_method' => 'automatic',
        'confirm' => true,
        'shipping' => [
            'name' => $request->customerName,
            'address' => [
                'line1' => 'UAE',
                'country' => 'AE',
            ],
        ],
        'return_url' => route('stripe.return'), // Specify the return URL here 
    ]);



    $paymentMethodId = $paymentIntent->payment_method;
    
    $user = auth()->user();

    $user_plan = User::find($user->id);
    
    $lastInvoice = Invoice::orderBy('id', 'desc')->first();

    // If there is no previous invoice, start with 'IN-01'
    if (!$lastInvoice) {
        $newInvoiceId = 'IN-01';
    } else {
        // Extract the number part from the last invoice_id and increment it
        $lastIdNumber = intval(str_replace('IN-', '', $lastInvoice->invoice_id));
        $newIdNumber = $lastIdNumber + 1;
    
        // Format the new ID with leading zeros if necessary (e.g., 'IN-01', 'IN-002')
        $newInvoiceId = 'IN-' . str_pad($newIdNumber, 2, '0', STR_PAD_LEFT);
    }
    
    
    $Invoice = Invoice::create([
        'invoice_id' => $newInvoiceId,
        'transaction_id' => $paymentMethodId,
        'bank_id' => 0,
        'company_id' => $user->id,
        'company_image' => $user->avatar_img ? $user->avatar_img : 'not available',
        'company_name' => $user->first_name.' '.$user->last_name,
        'company_number' => $user->mobile_no,
        'plan_status' => 1,
        'plan_actived' => $plan->id,
        'plan_issued_date' => now(),
        'plan_paided_status' => 1,
        'plan_amount' => $plan->amount,
        'billing_type' => 1,
        'billing_start_date' => now(),
        'billing_end_date' => now(),
        'status' => 1,
        'created_by' => $user->id,
        'created_at' => now()
    ]);
    

    $Payment = Payment::create([
        'plan_id' => $plan->id,
        'amount' => $plan->amount,
        'customer_name' => $request->customerName,
        'paymentmethod_id' => $paymentMethodId,
    ]);



    if ($user_plan) {
        $user_plan->update(['plan_actived' => $id]); 

        $user_plan->plan_day_remain += $plan->days;
        $user_plan->save();
    }

    return response()->json([
        'success' => true,
        'clientSecret' => $paymentIntent->client_secret,
    ]);

} catch (\Exception $e) {
    return response()->json([
        'success' => false,
        'error' => $e->getMessage(), // Optionally, you can include the error message
    ]);
}
    }

    /**
     * Handle the return URL after payment confirmation.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function stripeReturn(): RedirectResponse
    {
        return redirect()->route('stripe')->with('success', 'Payment successful!');
    }
}
